import { PrismaClient } from '@prisma/client';
import bcrypt from 'bcryptjs';

const prisma = new PrismaClient();

async function main() {
  console.log('🌱 Seeding RepGuide database...');

  // Create admin user
  const adminPassword = await bcrypt.hash('Admin123!', 12);
  const admin = await prisma.user.upsert({
    where: { email: 'admin@repguide.com' },
    update: {},
    create: {
      email: 'admin@repguide.com',
      name: 'Admin',
      password: adminPassword,
      role: 'ADMIN',
      status: 'ACTIVE',
    },
  });
  console.log(`✅ Admin user created: ${admin.email}`);

  // Create categories
  const categories = [
    {
      name: 'Sneakers',
      slug: 'sneakers',
      description: 'High-quality replica sneakers from top sellers',
      icon: '👟',
      order: 1,
    },
    {
      name: 'Clothing',
      slug: 'clothing',
      description: 'Designer clothing replicas - shirts, hoodies, jackets',
      icon: '👕',
      order: 2,
    },
    {
      name: 'Bags',
      slug: 'bags',
      description: 'Luxury bag replicas - backpacks, shoulder bags, wallets',
      icon: '👜',
      order: 3,
    },
    {
      name: 'Watches',
      slug: 'watches',
      description: 'Premium watch replicas from trusted sellers',
      icon: '⌚',
      order: 4,
    },
    {
      name: 'Accessories',
      slug: 'accessories',
      description: 'Belts, jewelry, sunglasses and more',
      icon: '🕶️',
      order: 5,
    },
    {
      name: 'Electronics',
      slug: 'electronics',
      description: 'Tech gadgets and accessories',
      icon: '📱',
      order: 6,
    },
  ];

  for (const cat of categories) {
    await prisma.category.upsert({
      where: { slug: cat.slug },
      update: cat,
      create: cat,
    });
  }
  console.log(`✅ Categories created: ${categories.length}`);

  // Get all category IDs
  const sneakersCategory = await prisma.category.findUnique({ where: { slug: 'sneakers' } });
  const clothingCategory = await prisma.category.findUnique({ where: { slug: 'clothing' } });
  const bagsCategory = await prisma.category.findUnique({ where: { slug: 'bags' } });
  const watchesCategory = await prisma.category.findUnique({ where: { slug: 'watches' } });
  const accessoriesCategory = await prisma.category.findUnique({ where: { slug: 'accessories' } });
  const electronicsCategory = await prisma.category.findUnique({ where: { slug: 'electronics' } });

  // Create sample products for ALL categories
  const products = [
    // SNEAKERS
    {
      categoryId: sneakersCategory!.id,
      name: 'AJ1 Retro High OG "Chicago"',
      slug: 'aj1-retro-high-chicago',
      description: 'Classic Jordan 1 Chicago colorway. Premium materials and accurate details.',
      price: 450,
      currency: 'CNY',
      images: JSON.stringify(['https://images.unsplash.com/photo-1556906781-9a412961c28c?w=500']),
      thumbnail: 'https://images.unsplash.com/photo-1556906781-9a412961c28c?w=300',
      source: 'WEIDIAN',
      sourceUrl: 'https://weidian.com/item.html?itemID=1234567890',
      hipobuyLink: 'https://www.hipobuy.com/product/1234567890',
      shopName: 'Top Sneaker Store',
      tags: JSON.stringify(['jordan', 'aj1', 'chicago']),
      isFeatured: true,
      rating: 4.8,
      reviewCount: 156,
    },
    {
      categoryId: sneakersCategory!.id,
      name: 'Yeezy 350 V2 "Zebra"',
      slug: 'yeezy-350-v2-zebra',
      description: 'Adidas Yeezy Boost 350 V2 Zebra. Comfortable boost sole.',
      price: 380,
      currency: 'CNY',
      images: JSON.stringify(['https://images.unsplash.com/photo-1600185365483-26d7a4cc7519?w=500']),
      thumbnail: 'https://images.unsplash.com/photo-1600185365483-26d7a4cc7519?w=300',
      source: 'TAOBAO',
      sourceUrl: 'https://item.taobao.com/item.htm?id=987654321',
      hipobuyLink: 'https://www.hipobuy.com/product/987654321',
      shopName: 'Yeezy Factory',
      tags: JSON.stringify(['yeezy', 'adidas', '350']),
      isFeatured: true,
      rating: 4.6,
      reviewCount: 89,
    },
    {
      categoryId: sneakersCategory!.id,
      name: 'Nike Dunk Low "Panda"',
      slug: 'nike-dunk-low-panda',
      description: 'Classic black and white colorway. Clean and versatile.',
      price: 320,
      currency: 'CNY',
      images: JSON.stringify(['https://images.unsplash.com/photo-1549298916-b41d501d3772?w=500']),
      thumbnail: 'https://images.unsplash.com/photo-1549298916-b41d501d3772?w=300',
      source: 'WEIDIAN',
      sourceUrl: 'https://weidian.com/item.html?itemID=111111111',
      hipobuyLink: 'https://www.hipobuy.com/product/111111111',
      shopName: 'Dunk Master',
      tags: JSON.stringify(['nike', 'dunk', 'panda']),
      isFeatured: false,
      rating: 4.7,
      reviewCount: 203,
    },
    {
      categoryId: sneakersCategory!.id,
      name: 'New Balance 550 "White Green"',
      slug: 'nb-550-white-green',
      description: 'Retro basketball style. Premium leather upper.',
      price: 290,
      currency: 'CNY',
      images: JSON.stringify(['https://images.unsplash.com/photo-1539185441755-769473a23570?w=500']),
      thumbnail: 'https://images.unsplash.com/photo-1539185441755-769473a23570?w=300',
      source: 'TAOBAO',
      sourceUrl: 'https://item.taobao.com/item.htm?id=222222222',
      hipobuyLink: 'https://www.hipobuy.com/product/222222222',
      shopName: 'NB Store',
      tags: JSON.stringify(['new balance', '550', 'retro']),
      isFeatured: false,
      rating: 4.5,
      reviewCount: 78,
    },

    // CLOTHING
    {
      categoryId: clothingCategory!.id,
      name: 'Essentials Hoodie - Cream',
      slug: 'essentials-hoodie-cream',
      description: 'Fear of God Essentials hoodie in cream color. Heavyweight cotton.',
      price: 180,
      currency: 'CNY',
      images: JSON.stringify(['https://images.unsplash.com/photo-1556821840-3a63f95609a7?w=500']),
      thumbnail: 'https://images.unsplash.com/photo-1556821840-3a63f95609a7?w=300',
      source: 'TAOBAO',
      sourceUrl: 'https://item.taobao.com/item.htm?id=111222333',
      hipobuyLink: 'https://www.hipobuy.com/product/111222333',
      shopName: 'Essentials Store',
      tags: JSON.stringify(['essentials', 'fog', 'hoodie']),
      isFeatured: true,
      rating: 4.7,
      reviewCount: 234,
    },
    {
      categoryId: clothingCategory!.id,
      name: 'Supreme Box Logo Tee - Black',
      slug: 'supreme-box-logo-tee-black',
      description: 'Classic Supreme box logo t-shirt. 100% cotton.',
      price: 120,
      currency: 'CNY',
      images: JSON.stringify(['https://images.unsplash.com/photo-1576566588028-4147f3842f27?w=500']),
      thumbnail: 'https://images.unsplash.com/photo-1576566588028-4147f3842f27?w=300',
      source: 'TAOBAO',
      sourceUrl: 'https://item.taobao.com/item.htm?id=333444555',
      hipobuyLink: 'https://www.hipobuy.com/product/333444555',
      shopName: 'Street Store',
      tags: JSON.stringify(['supreme', 'box logo', 'tee']),
      isFeatured: false,
      rating: 4.4,
      reviewCount: 189,
    },
    {
      categoryId: clothingCategory!.id,
      name: 'TNF Nuptse Jacket - Black',
      slug: 'tnf-nuptse-jacket-black',
      description: 'The North Face 1996 Retro Nuptse. 700 fill down.',
      price: 380,
      currency: 'CNY',
      images: JSON.stringify(['https://images.unsplash.com/photo-1544022613-e87ca75a784a?w=500']),
      thumbnail: 'https://images.unsplash.com/photo-1544022613-e87ca75a784a?w=300',
      source: 'WEIDIAN',
      sourceUrl: 'https://weidian.com/item.html?itemID=666777888',
      hipobuyLink: 'https://www.hipobuy.com/product/666777888',
      shopName: 'Outdoor Gear',
      tags: JSON.stringify(['tnf', 'north face', 'nuptse', 'jacket']),
      isFeatured: true,
      rating: 4.8,
      reviewCount: 312,
    },
    {
      categoryId: clothingCategory!.id,
      name: 'Stussy Basic Hoodie - Grey',
      slug: 'stussy-hoodie-grey',
      description: 'Stussy embroidered logo hoodie. Soft fleece lining.',
      price: 160,
      currency: 'CNY',
      images: JSON.stringify(['https://images.unsplash.com/photo-1620799140408-edc6dcb6d633?w=500']),
      thumbnail: 'https://images.unsplash.com/photo-1620799140408-edc6dcb6d633?w=300',
      source: 'TAOBAO',
      sourceUrl: 'https://item.taobao.com/item.htm?id=444555666',
      hipobuyLink: 'https://www.hipobuy.com/product/444555666',
      shopName: 'Stussy Rep',
      tags: JSON.stringify(['stussy', 'hoodie', 'streetwear']),
      isFeatured: false,
      rating: 4.5,
      reviewCount: 145,
    },

    // BAGS
    {
      categoryId: bagsCategory!.id,
      name: 'LV Keepall 45 Bandoulière',
      slug: 'lv-keepall-45',
      description: 'Louis Vuitton Keepall travel bag. Monogram canvas.',
      price: 680,
      currency: 'CNY',
      images: JSON.stringify(['https://images.unsplash.com/photo-1548036328-c9fa89d128fa?w=500']),
      thumbnail: 'https://images.unsplash.com/photo-1548036328-c9fa89d128fa?w=300',
      source: 'WEIDIAN',
      sourceUrl: 'https://weidian.com/item.html?itemID=444555666',
      hipobuyLink: 'https://www.hipobuy.com/product/444555666',
      shopName: 'Luxury Bags Store',
      tags: JSON.stringify(['lv', 'louis vuitton', 'keepall']),
      isFeatured: true,
      rating: 4.9,
      reviewCount: 67,
    },
    {
      categoryId: bagsCategory!.id,
      name: 'Goyard Saint Louis PM',
      slug: 'goyard-saint-louis-pm',
      description: 'Goyard tote bag. Hand-painted Y pattern.',
      price: 520,
      currency: 'CNY',
      images: JSON.stringify(['https://images.unsplash.com/photo-1584917865442-de89df76afd3?w=500']),
      thumbnail: 'https://images.unsplash.com/photo-1584917865442-de89df76afd3?w=300',
      source: 'WEIDIAN',
      sourceUrl: 'https://weidian.com/item.html?itemID=555666777',
      hipobuyLink: 'https://www.hipobuy.com/product/555666777',
      shopName: 'Designer Bags',
      tags: JSON.stringify(['goyard', 'tote', 'saint louis']),
      isFeatured: false,
      rating: 4.7,
      reviewCount: 98,
    },
    {
      categoryId: bagsCategory!.id,
      name: 'LV Mini Backpack',
      slug: 'lv-mini-backpack',
      description: 'Louis Vuitton Palm Springs Mini. Monogram print.',
      price: 450,
      currency: 'CNY',
      images: JSON.stringify(['https://images.unsplash.com/photo-1553062407-98eeb64c6a62?w=500']),
      thumbnail: 'https://images.unsplash.com/photo-1553062407-98eeb64c6a62?w=300',
      source: 'TAOBAO',
      sourceUrl: 'https://item.taobao.com/item.htm?id=777888999',
      hipobuyLink: 'https://www.hipobuy.com/product/777888999',
      shopName: 'LV Factory',
      tags: JSON.stringify(['lv', 'backpack', 'mini']),
      isFeatured: false,
      rating: 4.6,
      reviewCount: 156,
    },
    {
      categoryId: bagsCategory!.id,
      name: 'Prada Re-Nylon Shoulder Bag',
      slug: 'prada-renylon-shoulder',
      description: 'Prada Re-Nylon crossbody bag. Recycled nylon.',
      price: 380,
      currency: 'CNY',
      images: JSON.stringify(['https://images.unsplash.com/photo-1590874103328-eac38a683ce7?w=500']),
      thumbnail: 'https://images.unsplash.com/photo-1590874103328-eac38a683ce7?w=300',
      source: 'WEIDIAN',
      sourceUrl: 'https://weidian.com/item.html?itemID=888999000',
      hipobuyLink: 'https://www.hipobuy.com/product/888999000',
      shopName: 'Prada Store',
      tags: JSON.stringify(['prada', 'nylon', 'shoulder bag']),
      isFeatured: false,
      rating: 4.5,
      reviewCount: 87,
    },

    // WATCHES
    {
      categoryId: watchesCategory!.id,
      name: 'Rolex Submariner Date - Black',
      slug: 'rolex-submariner-black',
      description: 'Classic Rolex Submariner. Automatic movement. Ceramic bezel.',
      price: 1680,
      currency: 'CNY',
      images: JSON.stringify(['https://images.unsplash.com/photo-1523275335684-37898b6baf30?w=500']),
      thumbnail: 'https://images.unsplash.com/photo-1523275335684-37898b6baf30?w=300',
      source: 'WEIDIAN',
      sourceUrl: 'https://weidian.com/item.html?itemID=111222333',
      hipobuyLink: 'https://www.hipobuy.com/product/w111222333',
      shopName: 'Watch Factory',
      tags: JSON.stringify(['rolex', 'submariner', 'automatic']),
      isFeatured: true,
      rating: 4.9,
      reviewCount: 234,
    },
    {
      categoryId: watchesCategory!.id,
      name: 'Omega Seamaster 300M',
      slug: 'omega-seamaster-300m',
      description: 'Omega Seamaster diver watch. Blue dial, wave pattern.',
      price: 1480,
      currency: 'CNY',
      images: JSON.stringify(['https://images.unsplash.com/photo-1522312346375-d1a52e2b99b3?w=500']),
      thumbnail: 'https://images.unsplash.com/photo-1522312346375-d1a52e2b99b3?w=300',
      source: 'WEIDIAN',
      sourceUrl: 'https://weidian.com/item.html?itemID=222333444',
      hipobuyLink: 'https://www.hipobuy.com/product/w222333444',
      shopName: 'Omega Rep',
      tags: JSON.stringify(['omega', 'seamaster', 'diver']),
      isFeatured: false,
      rating: 4.8,
      reviewCount: 178,
    },
    {
      categoryId: watchesCategory!.id,
      name: 'AP Royal Oak - Blue Dial',
      slug: 'ap-royal-oak-blue',
      description: 'Audemars Piguet Royal Oak 41mm. Blue tapisserie dial.',
      price: 1880,
      currency: 'CNY',
      images: JSON.stringify(['https://images.unsplash.com/photo-1587836374828-4dbafa94cf0e?w=500']),
      thumbnail: 'https://images.unsplash.com/photo-1587836374828-4dbafa94cf0e?w=300',
      source: 'WEIDIAN',
      sourceUrl: 'https://weidian.com/item.html?itemID=333444555',
      hipobuyLink: 'https://www.hipobuy.com/product/w333444555',
      shopName: 'AP Factory',
      tags: JSON.stringify(['ap', 'royal oak', 'luxury']),
      isFeatured: false,
      rating: 4.7,
      reviewCount: 145,
    },
    {
      categoryId: watchesCategory!.id,
      name: 'Cartier Santos Medium',
      slug: 'cartier-santos-medium',
      description: 'Cartier Santos de Cartier. Stainless steel bracelet.',
      price: 1280,
      currency: 'CNY',
      images: JSON.stringify(['https://images.unsplash.com/photo-1548171915-e79a380a2a4b?w=500']),
      thumbnail: 'https://images.unsplash.com/photo-1548171915-e79a380a2a4b?w=300',
      source: 'TAOBAO',
      sourceUrl: 'https://item.taobao.com/item.htm?id=444555666',
      hipobuyLink: 'https://www.hipobuy.com/product/w444555666',
      shopName: 'Cartier Store',
      tags: JSON.stringify(['cartier', 'santos', 'dress watch']),
      isFeatured: false,
      rating: 4.6,
      reviewCount: 89,
    },

    // ACCESSORIES
    {
      categoryId: accessoriesCategory!.id,
      name: 'LV Initiales Belt - Black',
      slug: 'lv-initiales-belt-black',
      description: 'Louis Vuitton reversible belt. Monogram to plain black.',
      price: 280,
      currency: 'CNY',
      images: JSON.stringify(['https://images.unsplash.com/photo-1553062407-98eeb64c6a62?w=500']),
      thumbnail: 'https://images.unsplash.com/photo-1553062407-98eeb64c6a62?w=300',
      source: 'TAOBAO',
      sourceUrl: 'https://item.taobao.com/item.htm?id=555666777',
      hipobuyLink: 'https://www.hipobuy.com/product/a555666777',
      shopName: 'Belt Master',
      tags: JSON.stringify(['lv', 'belt', 'leather']),
      isFeatured: true,
      rating: 4.6,
      reviewCount: 312,
    },
    {
      categoryId: accessoriesCategory!.id,
      name: 'Ray-Ban Aviator Classic',
      slug: 'rayban-aviator-classic',
      description: 'Ray-Ban Aviator sunglasses. Gold frame, green lens.',
      price: 120,
      currency: 'CNY',
      images: JSON.stringify(['https://images.unsplash.com/photo-1572635196237-14b3f281503f?w=500']),
      thumbnail: 'https://images.unsplash.com/photo-1572635196237-14b3f281503f?w=300',
      source: 'TAOBAO',
      sourceUrl: 'https://item.taobao.com/item.htm?id=666777888',
      hipobuyLink: 'https://www.hipobuy.com/product/a666777888',
      shopName: 'Sunglasses Shop',
      tags: JSON.stringify(['rayban', 'aviator', 'sunglasses']),
      isFeatured: false,
      rating: 4.5,
      reviewCount: 245,
    },
    {
      categoryId: accessoriesCategory!.id,
      name: 'Gucci GG Marmont Belt',
      slug: 'gucci-gg-marmont-belt',
      description: 'Gucci double G buckle belt. Black leather.',
      price: 320,
      currency: 'CNY',
      images: JSON.stringify(['https://images.unsplash.com/photo-1594534475808-b18fc33b045e?w=500']),
      thumbnail: 'https://images.unsplash.com/photo-1594534475808-b18fc33b045e?w=300',
      source: 'WEIDIAN',
      sourceUrl: 'https://weidian.com/item.html?itemID=777888999',
      hipobuyLink: 'https://www.hipobuy.com/product/a777888999',
      shopName: 'Gucci Acc',
      tags: JSON.stringify(['gucci', 'belt', 'gg']),
      isFeatured: false,
      rating: 4.7,
      reviewCount: 178,
    },
    {
      categoryId: accessoriesCategory!.id,
      name: 'Chrome Hearts Ring - Silver',
      slug: 'chrome-hearts-ring',
      description: 'Chrome Hearts cross ring. 925 sterling silver.',
      price: 180,
      currency: 'CNY',
      images: JSON.stringify(['https://images.unsplash.com/photo-1605100804763-247f67b3557e?w=500']),
      thumbnail: 'https://images.unsplash.com/photo-1605100804763-247f67b3557e?w=300',
      source: 'TAOBAO',
      sourceUrl: 'https://item.taobao.com/item.htm?id=888999111',
      hipobuyLink: 'https://www.hipobuy.com/product/a888999111',
      shopName: 'CH Jewelry',
      tags: JSON.stringify(['chrome hearts', 'ring', 'silver']),
      isFeatured: false,
      rating: 4.4,
      reviewCount: 134,
    },

    // ELECTRONICS
    {
      categoryId: electronicsCategory!.id,
      name: 'AirPods Pro 2nd Gen',
      slug: 'airpods-pro-2',
      description: 'Apple AirPods Pro replica. Active noise cancellation.',
      price: 180,
      currency: 'CNY',
      images: JSON.stringify(['https://images.unsplash.com/photo-1600294037681-c80b4cb5b434?w=500']),
      thumbnail: 'https://images.unsplash.com/photo-1600294037681-c80b4cb5b434?w=300',
      source: 'TAOBAO',
      sourceUrl: 'https://item.taobao.com/item.htm?id=999111222',
      hipobuyLink: 'https://www.hipobuy.com/product/e999111222',
      shopName: 'Tech Store',
      tags: JSON.stringify(['airpods', 'apple', 'earbuds']),
      isFeatured: true,
      rating: 4.3,
      reviewCount: 456,
    },
    {
      categoryId: electronicsCategory!.id,
      name: 'Apple Watch Ultra Band',
      slug: 'apple-watch-ultra-band',
      description: 'Alpine loop band for Apple Watch. Multiple colors.',
      price: 45,
      currency: 'CNY',
      images: JSON.stringify(['https://images.unsplash.com/photo-1434493789847-2f02dc6ca35d?w=500']),
      thumbnail: 'https://images.unsplash.com/photo-1434493789847-2f02dc6ca35d?w=300',
      source: 'TAOBAO',
      sourceUrl: 'https://item.taobao.com/item.htm?id=111222333',
      hipobuyLink: 'https://www.hipobuy.com/product/e111222333',
      shopName: 'Watch Bands',
      tags: JSON.stringify(['apple watch', 'band', 'strap']),
      isFeatured: false,
      rating: 4.5,
      reviewCount: 567,
    },
    {
      categoryId: electronicsCategory!.id,
      name: 'MagSafe Charger Stand',
      slug: 'magsafe-charger-stand',
      description: '3-in-1 MagSafe charging station. iPhone, Watch, AirPods.',
      price: 120,
      currency: 'CNY',
      images: JSON.stringify(['https://images.unsplash.com/photo-1586816879360-004f5b0c51e5?w=500']),
      thumbnail: 'https://images.unsplash.com/photo-1586816879360-004f5b0c51e5?w=300',
      source: 'TAOBAO',
      sourceUrl: 'https://item.taobao.com/item.htm?id=222333444',
      hipobuyLink: 'https://www.hipobuy.com/product/e222333444',
      shopName: 'Charger Pro',
      tags: JSON.stringify(['magsafe', 'charger', 'wireless']),
      isFeatured: false,
      rating: 4.4,
      reviewCount: 289,
    },
    {
      categoryId: electronicsCategory!.id,
      name: 'Mechanical Keyboard - RGB',
      slug: 'mechanical-keyboard-rgb',
      description: 'Wireless mechanical keyboard. Hot-swappable switches.',
      price: 280,
      currency: 'CNY',
      images: JSON.stringify(['https://images.unsplash.com/photo-1595225476474-87563907a212?w=500']),
      thumbnail: 'https://images.unsplash.com/photo-1595225476474-87563907a212?w=300',
      source: 'TAOBAO',
      sourceUrl: 'https://item.taobao.com/item.htm?id=333444555',
      hipobuyLink: 'https://www.hipobuy.com/product/e333444555',
      shopName: 'Keyboard Master',
      tags: JSON.stringify(['keyboard', 'mechanical', 'gaming']),
      isFeatured: false,
      rating: 4.6,
      reviewCount: 198,
    },
  ];

  for (const product of products) {
    await prisma.product.upsert({
      where: { slug: product.slug },
      update: product,
      create: product,
    });
  }
  console.log(`✅ Products created: ${products.length}`);

  console.log('🎉 Seeding completed!');
  console.log('\n📋 Admin Credentials:');
  console.log('  Email: admin@repguide.com');
  console.log('  Password: Admin123!');
}

main()
  .catch((e) => {
    console.error('❌ Seeding failed:', e);
    process.exit(1);
  })
  .finally(async () => {
    await prisma.$disconnect();
  });
